// MyChef
// Version: 1.0
// Author: Michael Baccia

const fs = require('fs');
const path = require('path');
const chef = require("cyberchef-node");
const express = require('express');
const favicon = require('serve-favicon');
const morgan = require("morgan");

const app = express();

// CLI argument handling
const { parseArgs } = require('node:util');
const options = {
    port: {
      type: 'string',
      short: 'p',
    }
};
let {values} = parseArgs({options, allowPositionals: false});

// Parse config
const config = require('/etc/mychef/mychef.json');

// Middleware
app.use(morgan('short'));
app.use(favicon(`/usr/share/mychef/favicon.ico`));

// Allow accessing files directly
app.use(express.static(config.root_dir));

// Apply CyberChef recipes
// https://gchq.github.io/CyberChef/
app.get('/chef/:recipe*', (req, res) => {
    const filepath = path.normalize(config.root_dir + req.params[0]);
    const recipe_path = `/etc/mychef/recipes/${req.params['recipe']}.json`;

    if (!filepath.startsWith(config.root_dir)) {
        console.log(`Attempted file access outside of root directory. Filepath: ${filepath}`);
        res.status(404);
        res.send('Not found');
        return;
    }

    if (!recipe_path.startsWith('/etc/mychef/recipes')) {
        console.log(`Attempted file access outside of recipe directory. Filepath: ${recipe_path}`);
        res.status(404);
        res.send('Not found');
        return;
    }

    let recipe;

    try {
        recipe = JSON.parse(fs.readFileSync(recipe_path, 'utf8'));
    } catch (err) {
        res.status(404);
        res.send('Invalid recipe');
        return;
    }

    try {
        const data = fs.readFileSync(filepath);
        res.status(200);
        res.send(chef.bake(data, recipe).toString());
        return;
    } catch (err) {
        res.status(404);
        res.send('Not found');
        return;
    }
});

const port = values['port'] ? values['port'] : config.port;
app.listen(Number(port), () => {
    console.log(`Express serving "${config.root_dir}" on port ${port}`);
})
